# backend/owner_panel/services/admin_service.py
from typing import List, Optional, Dict, Any
from sqlalchemy.orm import Session
from sqlalchemy import and_, or_
from datetime import datetime, timedelta
import logging

from core.models import Admin, TenantRegistry, RechargeHistory
from core.exceptions import NotFoundError, ValidationError
from core.utils import security, id_generator, datetime_utils
from admin_panel.services.tenant_service import TenantService

logger = logging.getLogger(__name__)

class AdminService:
    """Service for admin management operations"""
    
    def __init__(self):
        self.tenant_service = TenantService()
    
    async def create_admin(self, admin_data: Dict[str, Any], owner_id: int, db: Session) -> Admin:
        """Create new admin with complete setup"""
        # Validate data
        self._validate_admin_data(admin_data)
        
        # Check uniqueness
        existing = db.query(Admin).filter(
            or_(
                Admin.telegram_id == admin_data['telegram_id'],
                Admin.admin_username == admin_data['admin_username'],
                Admin.bot_username == admin_data['bot_username'],
                Admin.bot_token == admin_data['bot_token']
            )
        ).first()
        
        if existing:
            raise ValidationError("Admin with this Telegram ID, username, bot username, or bot token already exists")
        
        # Generate reserve ID
        admin_count = db.query(Admin).count() + 1
        reserve_id = id_generator.generate_reserve_id(
            admin_data.get('prefix', 'BOT'), 
            admin_count
        )
        
        # Create admin
        admin = Admin(
            owner_id=owner_id,
            telegram_id=admin_data['telegram_id'],
            admin_username=admin_data['admin_username'],
            admin_first_name=admin_data.get('first_name'),
            admin_last_name=admin_data.get('last_name'),
            email=admin_data.get('email'),
            password_hash=security.get_password_hash(admin_data['password']),
            reserve_id=reserve_id,
            bot_name=admin_data['bot_name'],
            bot_username=admin_data['bot_username'],
            bot_token=admin_data['bot_token'],
            server_name=admin_data.get('server_name'),
            server_ip=admin_data.get('server_ip'),
            domain_name=admin_data.get('domain_name'),
            payment_gateway=admin_data.get('payment_gateway', 'razorpay'),
            payment_api_key=admin_data.get('payment_api_key'),
            payment_api_secret=admin_data.get('payment_api_secret'),
            payment_merchant_id=admin_data.get('payment_merchant_id'),
            payment_currency=admin_data.get('payment_currency', 'INR'),
            recharge_status='inactive',
            monthly_key_limit=admin_data.get('monthly_key_limit', 1000),
            status='active',
            settings=admin_data.get('settings', {})
        )
        
        db.add(admin)
        db.flush()
        
        # Create tenant database
        await self.tenant_service.create_tenant(admin.id, reserve_id, db)
        
        # Create tenant registry
        tenant_registry = TenantRegistry(
            admin_id=admin.id,
            reserve_id=reserve_id,
            database_name=f"tenant_{reserve_id.lower()}",
            database_host=admin_data.get('database_host', 'localhost'),
            database_port=admin_data.get('database_port', 5432),
            database_user=admin_data.get('database_user', f"user_{reserve_id.lower()}"),
            database_password=security.generate_api_secret(),
            users_table=f"users_{reserve_id.lower()}",
            keys_table=f"keys_{reserve_id.lower()}",
            transactions_table=f"transactions_{reserve_id.lower()}",
            key_rules_table=f"key_rules_{reserve_id.lower()}",
            key_batches_table=f"key_batches_{reserve_id.lower()}",
            apk_versions_table=f"apk_versions_{reserve_id.lower()}",
            broadcasts_table=f"broadcasts_{reserve_id.lower()}",
            user_activity_table=f"user_activity_{reserve_id.lower()}",
            user_sessions_table=f"user_sessions_{reserve_id.lower()}",
            referrals_table=f"referrals_{reserve_id.lower()}",
            tickets_table=f"tickets_{reserve_id.lower()}",
            key_alert_config_table=f"key_alerts_{reserve_id.lower()}"
        )
        
        db.add(tenant_registry)
        db.commit()
        db.refresh(admin)
        
        logger.info(f"Admin created successfully: {admin.admin_username} (ID: {admin.id})")
        return admin
    
    async def update_admin(self, admin_id: int, update_data: Dict[str, Any], db: Session) -> Admin:
        """Update admin information"""
        admin = db.query(Admin).filter(Admin.id == admin_id).first()
        if not admin:
            raise NotFoundError(f"Admin {admin_id} not found")
        
        # Update fields
        for key, value in update_data.items():
            if hasattr(admin, key) and value is not None:
                if key == 'password':
                    setattr(admin, 'password_hash', security.get_password_hash(value))
                else:
                    setattr(admin, key, value)
        
        db.commit()
        db.refresh(admin)
        
        logger.info(f"Admin updated: {admin.admin_username} (ID: {admin.id})")
        return admin
    
    async def delete_admin(self, admin_id: int, db: Session) -> bool:
        """Delete admin and all associated data"""
        admin = db.query(Admin).filter(Admin.id == admin_id).first()
        if not admin:
            raise NotFoundError(f"Admin {admin_id} not found")
        
        # Delete tenant database
        await self.tenant_service.delete_tenant(admin.reserve_id)
        
        # Delete admin
        db.delete(admin)
        db.commit()
        
        logger.info(f"Admin deleted: {admin.admin_username} (ID: {admin.id})")
        return True
    
    async def get_admin_stats(self, admin_id: int, db: Session) -> Dict[str, Any]:
        """Get admin statistics"""
        admin = db.query(Admin).filter(Admin.id == admin_id).first()
        if not admin:
            raise NotFoundError(f"Admin {admin_id} not found")
        
        # Get tenant stats
        tenant_stats = await self.tenant_service.get_stats(admin.reserve_id)
        
        # Get recharge history
        recharges = db.query(RechargeHistory).filter(
            RechargeHistory.admin_id == admin_id
        ).all()
        
        total_recharged = sum(r.amount for r in recharges)
        last_recharge = recharges[-1] if recharges else None
        
        return {
            'admin_id': admin.id,
            'username': admin.admin_username,
            'reserve_id': admin.reserve_id,
            'bot_name': admin.bot_name,
            'status': admin.status,
            'recharge': {
                'status': admin.recharge_status,
                'plan': admin.recharge_plan,
                'expiry': admin.recharge_expiry.isoformat() if admin.recharge_expiry else None,
                'total_recharged': float(total_recharged),
                'last_recharge': {
                    'amount': float(last_recharge.amount) if last_recharge else None,
                    'date': last_recharge.created_at.isoformat() if last_recharge else None
                }
            },
            'tenant_stats': tenant_stats,
            'created_at': admin.created_at.isoformat(),
            'last_login': admin.last_login.isoformat() if admin.last_login else None
        }
    
    async def get_all_admins_summary(self, db: Session) -> List[Dict[str, Any]]:
        """Get summary of all admins"""
        admins = db.query(Admin).all()
        summary = []
        
        for admin in admins:
            # Get tenant stats
            tenant_stats = await self.tenant_service.get_basic_stats(admin.reserve_id)
            
            summary.append({
                'id': admin.id,
                'username': admin.admin_username,
                'reserve_id': admin.reserve_id,
                'bot_name': admin.bot_name,
                'status': admin.status,
                'recharge_status': admin.recharge_status,
                'recharge_expiry': admin.recharge_expiry.isoformat() if admin.recharge_expiry else None,
                'total_users': tenant_stats.get('total_users', 0),
                'total_revenue': tenant_stats.get('total_revenue', 0),
                'created_at': admin.created_at.isoformat()
            })
        
        return summary
    
    def _validate_admin_data(self, data: Dict[str, Any]):
        """Validate admin creation data"""
        required_fields = ['telegram_id', 'admin_username', 'password', 
                          'bot_name', 'bot_username', 'bot_token']
        
        for field in required_fields:
            if field not in data or not data[field]:
                raise ValidationError(f"Missing required field: {field}")
        
        # Validate telegram ID
        try:
            int(data['telegram_id'])
        except:
            raise ValidationError("Invalid telegram ID")
        
        # Validate bot token format
        if not data['bot_token'].count(':') == 1:
            raise ValidationError("Invalid bot token format")