# backend/owner_panel/services/owner_service.py
from typing import Dict, Any, Optional
from sqlalchemy.orm import Session
import logging

from core.models import Owner, SystemConfig, AuditLog
from core.exceptions import NotFoundError, ValidationError
from core.utils import security, datetime_utils

logger = logging.getLogger(__name__)

class OwnerService:
    """Service for owner account management"""
    
    async def get_owner_profile(self, owner_id: int, db: Session) -> Dict[str, Any]:
        """Get owner profile"""
        owner = db.query(Owner).filter(Owner.id == owner_id).first()
        if not owner:
            raise NotFoundError(f"Owner {owner_id} not found")
        
        return {
            'id': owner.id,
            'telegram_id': owner.telegram_id,
            'username': owner.username,
            'first_name': owner.first_name,
            'last_name': owner.last_name,
            'email': owner.email,
            'two_factor_enabled': owner.two_factor_enabled,
            'language': owner.language,
            'created_at': owner.created_at.isoformat(),
            'last_login': owner.last_login.isoformat() if owner.last_login else None
        }
    
    async def update_owner_profile(self, owner_id: int, update_data: Dict[str, Any], db: Session) -> Dict[str, Any]:
        """Update owner profile"""
        owner = db.query(Owner).filter(Owner.id == owner_id).first()
        if not owner:
            raise NotFoundError(f"Owner {owner_id} not found")
        
        # Update fields
        updatable_fields = ['first_name', 'last_name', 'email', 'language']
        for field in updatable_fields:
            if field in update_data:
                setattr(owner, field, update_data[field])
        
        db.commit()
        db.refresh(owner)
        
        logger.info(f"Owner profile updated: {owner.username}")
        
        return await self.get_owner_profile(owner_id, db)
    
    async def change_password(self, owner_id: int, old_password: str, new_password: str, db: Session) -> bool:
        """Change owner password"""
        owner = db.query(Owner).filter(Owner.id == owner_id).first()
        if not owner:
            raise NotFoundError(f"Owner {owner_id} not found")
        
        # Verify old password
        if not security.verify_password(old_password, owner.password_hash):
            raise ValidationError("Current password is incorrect")
        
        # Update password
        owner.password_hash = security.get_password_hash(new_password)
        db.commit()
        
        logger.info(f"Password changed for owner {owner.username}")
        return True
    
    async def enable_2fa(self, owner_id: int, db: Session) -> Dict[str, Any]:
        """Enable two-factor authentication"""
        owner = db.query(Owner).filter(Owner.id == owner_id).first()
        if not owner:
            raise NotFoundError(f"Owner {owner_id} not found")
        
        # Generate 2FA secret
        import pyotp
        secret = pyotp.random_base32()
        
        owner.two_factor_secret = secret
        owner.two_factor_enabled = True
        db.commit()
        
        # Generate provisioning URI
        totp = pyotp.TOTP(secret)
        provisioning_uri = totp.provisioning_uri(
            owner.username,
            issuer_name="Telegram Bot System"
        )
        
        logger.info(f"2FA enabled for owner {owner.username}")
        
        return {
            'secret': secret,
            'provisioning_uri': provisioning_uri,
            'enabled': True
        }
    
    async def disable_2fa(self, owner_id: int, db: Session) -> bool:
        """Disable two-factor authentication"""
        owner = db.query(Owner).filter(Owner.id == owner_id).first()
        if not owner:
            raise NotFoundError(f"Owner {owner_id} not found")
        
        owner.two_factor_secret = None
        owner.two_factor_enabled = False
        db.commit()
        
        logger.info(f"2FA disabled for owner {owner.username}")
        return True
    
    async def verify_2fa(self, owner_id: int, code: str, db: Session) -> bool:
        """Verify 2FA code"""
        owner = db.query(Owner).filter(Owner.id == owner_id).first()
        if not owner or not owner.two_factor_secret:
            return False
        
        import pyotp
        totp = pyotp.TOTP(owner.two_factor_secret)
        return totp.verify(code)
    
    async def get_notification_preferences(self, owner_id: int, db: Session) -> Dict[str, Any]:
        """Get owner notification preferences"""
        owner = db.query(Owner).filter(Owner.id == owner_id).first()
        if not owner:
            raise NotFoundError(f"Owner {owner_id} not found")
        
        return owner.notification_preferences or {
            'email_alerts': True,
            'telegram_alerts': True,
            'daily_report': True,
            'weekly_report': True,
            'critical_alerts': True
        }
    
    async def update_notification_preferences(self, owner_id: int, preferences: Dict[str, Any], db: Session) -> Dict[str, Any]:
        """Update notification preferences"""
        owner = db.query(Owner).filter(Owner.id == owner_id).first()
        if not owner:
            raise NotFoundError(f"Owner {owner_id} not found")
        
        owner.notification_preferences = preferences
        db.commit()
        
        logger.info(f"Notification preferences updated for owner {owner.username}")
        return preferences