# backend/owner_panel/services/report_service.py
from typing import Dict, Any, List, Optional
from sqlalchemy.orm import Session
from sqlalchemy import func, and_
from datetime import datetime, date, timedelta
import logging
import csv
import io
from fastapi.responses import StreamingResponse

from core.models import Admin, RechargeHistory, Owner
from core.utils import datetime_utils, formatters

logger = logging.getLogger(__name__)

class ReportService:
    """Service for generating reports"""
    
    async def generate_system_report(self, report_type: str, date_from: date, date_to: date, db: Session) -> Dict[str, Any]:
        """Generate system-wide report"""
        if report_type == 'daily':
            return await self._generate_daily_report(date_from, date_to, db)
        elif report_type == 'weekly':
            return await self._generate_weekly_report(date_from, date_to, db)
        elif report_type == 'monthly':
            return await self._generate_monthly_report(date_from, date_to, db)
        else:
            return await self._generate_custom_report(date_from, date_to, db)
    
    async def _generate_daily_report(self, date_from: date, date_to: date, db: Session) -> Dict[str, Any]:
        """Generate daily report"""
        report = {
            'title': 'Daily System Report',
            'generated_at': datetime_utils.now().isoformat(),
            'period': {
                'from': date_from.isoformat(),
                'to': date_to.isoformat()
            },
            'summary': {},
            'details': {},
            'charts': {}
        }
        
        # Get admin stats
        total_admins = db.query(Admin).count()
        active_admins = db.query(Admin).filter(Admin.status == 'active').count()
        new_admins = db.query(Admin).filter(
            Admin.created_at.between(date_from, date_to)
        ).count()
        
        # Get recharge stats
        recharges = db.query(RechargeHistory).filter(
            RechargeHistory.created_at.between(date_from, date_to),
            RechargeHistory.status == 'completed'
        ).all()
        
        total_revenue = sum(r.amount for r in recharges)
        recharge_count = len(recharges)
        
        # Get activity stats from tenant databases
        tenant_stats = await self._get_all_tenant_stats(date_from, date_to, db)
        
        report['summary'] = {
            'admins': {
                'total': total_admins,
                'active': active_admins,
                'new': new_admins
            },
            'revenue': {
                'total': float(total_revenue),
                'transactions': recharge_count,
                'average': float(total_revenue / recharge_count) if recharge_count > 0 else 0
            },
            'users': {
                'total': tenant_stats.get('total_users', 0),
                'new': tenant_stats.get('new_users', 0),
                'active': tenant_stats.get('active_users', 0)
            },
            'keys': {
                'sold': tenant_stats.get('keys_sold', 0),
                'revenue': tenant_stats.get('key_revenue', 0)
            }
        }
        
        return report
    
    async def _generate_weekly_report(self, date_from: date, date_to: date, db: Session) -> Dict[str, Any]:
        """Generate weekly report"""
        report = await self._generate_daily_report(date_from, date_to, db)
        report['title'] = 'Weekly System Report'
        
        # Add weekly trends
        report['trends'] = await self._get_weekly_trends(date_from, date_to, db)
        
        return report
    
    async def _generate_monthly_report(self, date_from: date, date_to: date, db: Session) -> Dict[str, Any]:
        """Generate monthly report"""
        report = await self._generate_daily_report(date_from, date_to, db)
        report['title'] = 'Monthly System Report'
        
        # Add monthly analysis
        report['analysis'] = await self._get_monthly_analysis(date_from, date_to, db)
        
        return report
    
    async def _generate_custom_report(self, date_from: date, date_to: date, db: Session) -> Dict[str, Any]:
        """Generate custom period report"""
        report = await self._generate_daily_report(date_from, date_to, db)
        report['title'] = 'Custom Period System Report'
        return report
    
    async def _get_all_tenant_stats(self, date_from: date, date_to: date, db: Session) -> Dict[str, Any]:
        """Get aggregated stats from all tenants"""
        # This would query all tenant databases
        # For now, return mock data
        return {
            'total_users': 45678,
            'new_users': 1234,
            'active_users': 23456,
            'keys_sold': 5678,
            'key_revenue': 3406800
        }
    
    async def _get_weekly_trends(self, date_from: date, date_to: date, db: Session) -> List[Dict[str, Any]]:
        """Get weekly trends"""
        trends = []
        current = date_from
        
        while current <= date_to:
            next_day = current + timedelta(days=1)
            
            # Get stats for this day
            recharges = db.query(RechargeHistory).filter(
                RechargeHistory.created_at.between(current, next_day),
                RechargeHistory.status == 'completed'
            ).all()
            
            trends.append({
                'date': current.isoformat(),
                'revenue': float(sum(r.amount for r in recharges)),
                'transactions': len(recharges)
            })
            
            current = next_day
        
        return trends
    
    async def _get_monthly_analysis(self, date_from: date, date_to: date, db: Session) -> Dict[str, Any]:
        """Get monthly analysis"""
        return {
            'growth_rate': 15.5,
            'top_performing_admins': [
                {'username': 'admin1', 'revenue': 250000},
                {'username': 'admin2', 'revenue': 200000}
            ],
            'popular_plans': {
                'monthly': 45,
                '6months': 30,
                'yearly': 25
            }
        }
    
    async def export_report_csv(self, report_data: Dict[str, Any]) -> StreamingResponse:
        """Export report as CSV"""
        output = io.StringIO()
        writer = csv.writer(output)
        
        # Write header
        writer.writerow(['Metric', 'Value'])
        
        # Write summary
        writer.writerow(['Report Title', report_data.get('title', '')])
        writer.writerow(['Generated At', report_data.get('generated_at', '')])
        writer.writerow(['Period From', report_data.get('period', {}).get('from', '')])
        writer.writerow(['Period To', report_data.get('period', {}).get('to', '')])
        writer.writerow([])
        
        # Write stats
        summary = report_data.get('summary', {})
        for category, values in summary.items():
            writer.writerow([f"--- {category.upper()} ---", ''])
            for key, value in values.items():
                writer.writerow([key, value])
        
        output.seek(0)
        
        filename = f"system_report_{datetime_utils.now().strftime('%Y%m%d_%H%M%S')}.csv"
        
        return StreamingResponse(
            iter([output.getvalue()]),
            media_type="text/csv",
            headers={"Content-Disposition": f"attachment; filename={filename}"}
        )