#!/usr/bin/env python3
# scripts/create_admin.py
import sys
import os
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

import asyncio
import getpass
import logging
from datetime import datetime, timedelta

from backend.core.database import SessionLocal
from backend.core.models import Owner, Admin
from backend.core.security import security
from backend.core.utils import id_generator

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

async def create_owner():
    """Create initial owner if not exists"""
    db = SessionLocal()
    try:
        owner = db.query(Owner).first()
        if owner:
            logger.info(f"Owner already exists: {owner.username}")
            return owner
        
        logger.info("Creating initial owner...")
        username = input("Enter owner username [owner]: ") or "owner"
        telegram_id = input("Enter owner telegram ID [123456789]: ") or "123456789"
        password = getpass.getpass("Enter owner password [ChangeMe123!]: ") or "ChangeMe123!"
        
        owner = Owner(
            telegram_id=int(telegram_id),
            username=username,
            first_name="System",
            last_name="Owner",
            password_hash=security.get_password_hash(password),
            two_factor_enabled=False
        )
        
        db.add(owner)
        db.commit()
        logger.info(f"Owner created successfully: {username}")
        return owner
        
    finally:
        db.close()

async def create_admin(owner):
    """Create initial admin"""
    db = SessionLocal()
    try:
        admin = db.query(Admin).first()
        if admin:
            logger.info(f"Admin already exists: {admin.admin_username}")
            return admin
        
        logger.info("Creating initial admin...")
        username = input("Enter admin username [admin]: ") or "admin"
        telegram_id = input("Enter admin telegram ID [987654321]: ") or "987654321"
        bot_name = input("Enter bot name [Test Bot]: ") or "Test Bot"
        bot_username = input("Enter bot username [@test_bot]: ") or "@test_bot"
        bot_token = input("Enter bot token: ") or "123456:ABCdefGHIjklMNOpqrsTUVwxyz"
        password = getpass.getpass("Enter admin password [Admin123!]: ") or "Admin123!"
        
        # Generate reserve ID
        admin_count = db.query(Admin).count() + 1
        reserve_id = id_generator.generate_reserve_id("TEST", admin_count)
        
        admin = Admin(
            owner_id=owner.id,
            telegram_id=int(telegram_id),
            admin_username=username,
            admin_first_name="Test",
            admin_last_name="Admin",
            password_hash=security.get_password_hash(password),
            reserve_id=reserve_id,
            bot_name=bot_name,
            bot_username=bot_username,
            bot_token=bot_token,
            payment_gateway="razorpay",
            payment_api_key="test_key",
            payment_api_secret="test_secret",
            recharge_status="active",
            recharge_expiry=datetime.utcnow() + timedelta(days=30),
            status="active"
        )
        
        db.add(admin)
        db.commit()
        logger.info(f"Admin created successfully: {username} with reserve ID: {reserve_id}")
        return admin
        
    finally:
        db.close()

async def main():
    logger.info("Creating initial users...")
    owner = await create_owner()
    if owner:
        await create_admin(owner)
    logger.info("Done!")

if __name__ == "__main__":
    asyncio.run(main())