# scripts/set_webhook.py
#!/usr/bin/env python3
"""
Script to set Telegram webhook
"""
import sys
import os
import asyncio
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from backend.config.settings import settings
import aiohttp
import logging

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

async def set_webhook():
    """Set webhook for Telegram bot"""
    if not settings.TELEGRAM_BOT_TOKEN:
        logger.error("TELEGRAM_BOT_TOKEN not set")
        return
    
    if not settings.TELEGRAM_WEBHOOK_URL:
        logger.error("TELEGRAM_WEBHOOK_URL not set")
        return
    
    url = f"https://api.telegram.org/bot{settings.TELEGRAM_BOT_TOKEN}/setWebhook"
    
    payload = {
        "url": f"{settings.TELEGRAM_WEBHOOK_URL}/webhook",
        "max_connections": settings.TELEGRAM_MAX_CONNECTIONS,
        "allowed_updates": ["message", "callback_query", "inline_query", "chosen_inline_result"]
    }
    
    if settings.TELEGRAM_WEBHOOK_SECRET:
        payload["secret_token"] = settings.TELEGRAM_WEBHOOK_SECRET
    
    async with aiohttp.ClientSession() as session:
        async with session.post(url, json=payload) as response:
            result = await response.json()
            if result.get("ok"):
                logger.info("Webhook set successfully")
                logger.info(f"Response: {result}")
            else:
                logger.error(f"Failed to set webhook: {result}")

async def get_webhook_info():
    """Get current webhook info"""
    if not settings.TELEGRAM_BOT_TOKEN:
        logger.error("TELEGRAM_BOT_TOKEN not set")
        return
    
    url = f"https://api.telegram.org/bot{settings.TELEGRAM_BOT_TOKEN}/getWebhookInfo"
    
    async with aiohttp.ClientSession() as session:
        async with session.get(url) as response:
            result = await response.json()
            if result.get("ok"):
                logger.info(f"Webhook info: {result}")
            else:
                logger.error(f"Failed to get webhook info: {result}")

async def delete_webhook():
    """Delete webhook"""
    if not settings.TELEGRAM_BOT_TOKEN:
        logger.error("TELEGRAM_BOT_TOKEN not set")
        return
    
    url = f"https://api.telegram.org/bot{settings.TELEGRAM_BOT_TOKEN}/deleteWebhook"
    
    async with aiohttp.ClientSession() as session:
        async with session.get(url) as response:
            result = await response.json()
            if result.get("ok"):
                logger.info("Webhook deleted successfully")
            else:
                logger.error(f"Failed to delete webhook: {result}")

if __name__ == "__main__":
    import argparse
    
    parser = argparse.ArgumentParser(description="Manage Telegram webhook")
    parser.add_argument("action", choices=["set", "get", "delete"], help="Action to perform")
    
    args = parser.parse_args()
    
    if args.action == "set":
        asyncio.run(set_webhook())
    elif args.action == "get":
        asyncio.run(get_webhook_info())
    elif args.action == "delete":
        asyncio.run(delete_webhook())