# backend/shared/constants/permissions.py
from enum import Enum

class UserRole(str, Enum):
    """User roles in the system"""
    OWNER = "owner"
    ADMIN = "admin"
    USER = "user"


class Permission(str, Enum):
    """Permission definitions"""
    
    # Owner permissions
    MANAGE_ADMINS = "manage_admins"
    VIEW_ALL_USERS = "view_all_users"
    VIEW_ALL_ACTIVITIES = "view_all_activities"
    MANAGE_RECHARGE = "manage_recharge"
    GLOBAL_BROADCAST = "global_broadcast"
    VIEW_SYSTEM_STATS = "view_system_stats"
    MANAGE_SETTINGS = "manage_settings"
    
    # Admin permissions
    VIEW_OWN_USERS = "view_own_users"
    VIEW_OWN_ACTIVITIES = "view_own_activities"
    MANAGE_KEYS = "manage_keys"
    MANAGE_PAYMENTS = "manage_payments"
    MANAGE_APK = "manage_apk"
    SEND_BROADCAST = "send_broadcast"
    VIEW_ANALYTICS = "view_analytics"
    
    # User permissions
    PURCHASE_KEYS = "purchase_keys"
    VIEW_OWN_KEYS = "view_own_keys"
    DOWNLOAD_APK = "download_apk"
    CONTACT_SUPPORT = "contact_support"


# Role to permissions mapping
ROLE_PERMISSIONS = {
    UserRole.OWNER: [
        Permission.MANAGE_ADMINS,
        Permission.VIEW_ALL_USERS,
        Permission.VIEW_ALL_ACTIVITIES,
        Permission.MANAGE_RECHARGE,
        Permission.GLOBAL_BROADCAST,
        Permission.VIEW_SYSTEM_STATS,
        Permission.MANAGE_SETTINGS,
    ],
    UserRole.ADMIN: [
        Permission.VIEW_OWN_USERS,
        Permission.VIEW_OWN_ACTIVITIES,
        Permission.MANAGE_KEYS,
        Permission.MANAGE_PAYMENTS,
        Permission.MANAGE_APK,
        Permission.SEND_BROADCAST,
        Permission.VIEW_ANALYTICS,
    ],
    UserRole.USER: [
        Permission.PURCHASE_KEYS,
        Permission.VIEW_OWN_KEYS,
        Permission.DOWNLOAD_APK,
        Permission.CONTACT_SUPPORT,
    ]
}


def has_permission(user_role: UserRole, permission: Permission) -> bool:
    """Check if role has specific permission"""
    return permission in ROLE_PERMISSIONS.get(user_role, [])


def get_role_permissions(role: UserRole) -> list:
    """Get all permissions for a role"""
    return ROLE_PERMISSIONS.get(role, [])