# backend/shared/decorators/role_required.py
from functools import wraps
from fastapi import HTTPException, Depends
from typing import Callable, List
import logging

from core.security import get_current_user
from shared.constants.permissions import UserRole

logger = logging.getLogger(__name__)

def role_required(allowed_roles: List[UserRole]):
    """Decorator to check if user has required role"""
    def decorator(func: Callable):
        @wraps(func)
        async def wrapper(*args, **kwargs):
            # Get current user from kwargs
            current_user = kwargs.get('current_user')
            if not current_user:
                # Try to get from request
                request = kwargs.get('request')
                if request:
                    current_user = await get_current_user(request)
            
            if not current_user:
                raise HTTPException(status_code=401, detail="Not authenticated")
            
            if current_user.role not in allowed_roles:
                logger.warning(f"User {current_user.id} with role {current_user.role} attempted to access restricted endpoint")
                raise HTTPException(status_code=403, detail="Insufficient permissions")
            
            return await func(*args, **kwargs)
        return wrapper
    return decorator


def owner_required(func: Callable):
    """Decorator for owner-only endpoints"""
    return role_required([UserRole.OWNER])(func)


def admin_required(func: Callable):
    """Decorator for admin-only endpoints"""
    return role_required([UserRole.OWNER, UserRole.ADMIN])(func)


def user_required(func: Callable):
    """Decorator for user endpoints"""
    return role_required([UserRole.OWNER, UserRole.ADMIN, UserRole.USER])(func)