# backend/shared/decorators/tenant_required.py
from functools import wraps
from fastapi import HTTPException, Depends
from typing import Callable
import logging

from core.database import tenant_db_manager
from core.models import Admin, TenantRegistry
from core.security import get_current_admin

logger = logging.getLogger(__name__)

def tenant_required(func: Callable):
    """Decorator to ensure tenant database is available and connected"""
    @wraps(func)
    async def wrapper(*args, **kwargs):
        # Get current admin
        current_admin = kwargs.get('current_admin')
        if not current_admin:
            raise HTTPException(status_code=401, detail="Not authenticated")
        
        # Check if admin has active tenant
        if not current_admin.reserve_id:
            raise HTTPException(status_code=400, detail="Admin has no tenant configured")
        
        # Check recharge status
        if current_admin.recharge_status != 'active':
            raise HTTPException(status_code=403, detail="Admin panel expired. Please recharge.")
        
        # Get tenant registry
        db = kwargs.get('db')
        if not db:
            raise HTTPException(status_code=500, detail="Database session not available")
        
        tenant = db.query(TenantRegistry).filter(
            TenantRegistry.admin_id == current_admin.id
        ).first()
        
        if not tenant:
            raise HTTPException(status_code=404, detail="Tenant not found")
        
        # Create tenant database URL
        db_url = f"postgresql://{tenant.database_user}:{tenant.database_password}@{tenant.database_host}:{tenant.database_port}/{tenant.database_name}"
        
        # Get tenant session
        try:
            tenant_session = tenant_db_manager.get_tenant_session(
                current_admin.reserve_id, db_url
            )
            kwargs['tenant_db'] = tenant_session
            kwargs['tenant_registry'] = tenant
        except Exception as e:
            logger.error(f"Failed to connect to tenant database: {e}")
            raise HTTPException(status_code=500, detail="Tenant database connection failed")
        
        return await func(*args, **kwargs)
    return wrapper