# backend/shared/inline_keyboards.py
"""
Shared inline keyboards for Telegram bot
These keyboards can be used across different bots
"""

from telegram import InlineKeyboardButton, InlineKeyboardMarkup
from typing import List, Dict, Optional


def get_main_menu_keyboard() -> InlineKeyboardMarkup:
    """Get main menu keyboard"""
    keyboard = [
        [
            InlineKeyboardButton("🛒 Buy Keys", callback_data="menu_buy"),
            InlineKeyboardButton("📋 View Plans", callback_data="menu_plans")
        ],
        [
            InlineKeyboardButton("🔑 My Keys", callback_data="menu_my_keys"),
            InlineKeyboardButton("📱 Download App", callback_data="menu_download")
        ],
        [
            InlineKeyboardButton("❓ Help", callback_data="menu_help"),
            InlineKeyboardButton("📧 Contact", callback_data="menu_contact")
        ]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_plans_keyboard() -> InlineKeyboardMarkup:
    """Get plans selection keyboard"""
    keyboard = [
        [
            InlineKeyboardButton("1 Day - ₹100", callback_data="plan_1d"),
            InlineKeyboardButton("3 Days - ₹250", callback_data="plan_3d")
        ],
        [
            InlineKeyboardButton("7 Days - ₹350", callback_data="plan_7d"),
            InlineKeyboardButton("30 Days - ₹600", callback_data="plan_30d")
        ],
        [
            InlineKeyboardButton("60 Days - ₹800", callback_data="plan_60d")
        ],
        [
            InlineKeyboardButton("🔙 Back to Menu", callback_data="back_to_menu")
        ]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_purchase_confirmation_keyboard() -> InlineKeyboardMarkup:
    """Get purchase confirmation keyboard"""
    keyboard = [
        [
            InlineKeyboardButton("✅ Confirm", callback_data="confirm_purchase"),
            InlineKeyboardButton("❌ Cancel", callback_data="cancel_purchase")
        ]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_payment_keyboard(payment_link: str) -> InlineKeyboardMarkup:
    """Get payment keyboard with payment link"""
    keyboard = [
        [InlineKeyboardButton("💳 Pay Now", url=payment_link)],
        [InlineKeyboardButton("✅ I've Paid", callback_data="check_payment")],
        [InlineKeyboardButton("❌ Cancel", callback_data="cancel_payment")]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_back_keyboard() -> InlineKeyboardMarkup:
    """Get back button only"""
    keyboard = [
        [InlineKeyboardButton("🔙 Back", callback_data="back")]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_back_to_menu_keyboard() -> InlineKeyboardMarkup:
    """Get back to menu button"""
    keyboard = [
        [InlineKeyboardButton("🔙 Back to Menu", callback_data="back_to_menu")]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_confirmation_keyboard() -> InlineKeyboardMarkup:
    """Get generic confirmation keyboard"""
    keyboard = [
        [
            InlineKeyboardButton("✅ Yes", callback_data="confirm_yes"),
            InlineKeyboardButton("❌ No", callback_data="confirm_no")
        ]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_cancel_keyboard() -> InlineKeyboardMarkup:
    """Get cancel button only"""
    keyboard = [
        [InlineKeyboardButton("❌ Cancel", callback_data="cancel")]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_my_keys_keyboard() -> InlineKeyboardMarkup:
    """Get my keys action keyboard"""
    keyboard = [
        [
            InlineKeyboardButton("🔄 Refresh", callback_data="refresh_keys"),
            InlineKeyboardButton("🛒 Buy More", callback_data="buy_more")
        ],
        [
            InlineKeyboardButton("📱 Download App", callback_data="download_app"),
            InlineKeyboardButton("🔙 Menu", callback_data="back_to_menu")
        ]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_extend_keyboard(keys: List[Dict]) -> InlineKeyboardMarkup:
    """Get key extension selection keyboard"""
    keyboard = []
    for key in keys:
        keyboard.append([
            InlineKeyboardButton(
                f"{key['key'][:8]}... ({key['days_left']} days left)",
                callback_data=f"extend_key_{key['id']}"
            )
        ])
    keyboard.append([InlineKeyboardButton("❌ Cancel", callback_data="cancel")])
    return InlineKeyboardMarkup(keyboard)


def get_extend_duration_keyboard() -> InlineKeyboardMarkup:
    """Get extension duration selection keyboard"""
    keyboard = [
        [
            InlineKeyboardButton("7 Days - ₹350", callback_data="extend_7d"),
            InlineKeyboardButton("30 Days - ₹600", callback_data="extend_30d")
        ],
        [
            InlineKeyboardButton("60 Days - ₹800", callback_data="extend_60d")
        ],
        [InlineKeyboardButton("❌ Cancel", callback_data="cancel")]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_download_keyboard(download_link: str) -> InlineKeyboardMarkup:
    """Get download keyboard with download link"""
    keyboard = [
        [InlineKeyboardButton("📥 Download Now", url=download_link)],
        [InlineKeyboardButton("🔙 Back to Menu", callback_data="back_to_menu")]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_help_keyboard() -> InlineKeyboardMarkup:
    """Get help menu keyboard"""
    keyboard = [
        [
            InlineKeyboardButton("📋 Plans", callback_data="menu_plans"),
            InlineKeyboardButton("🛒 Buy", callback_data="menu_buy")
        ],
        [
            InlineKeyboardButton("🔑 My Keys", callback_data="menu_my_keys"),
            InlineKeyboardButton("📱 Download", callback_data="menu_download")
        ],
        [
            InlineKeyboardButton("📧 Contact Support", url="https://t.me/support_bot")
        ]
    ]
    return InlineKeyboardMarkup(keyboard)


def get_pagination_keyboard(current_page: int, total_pages: int, prefix: str) -> InlineKeyboardMarkup:
    """Get pagination keyboard"""
    keyboard = []
    
    if total_pages > 1:
        row = []
        if current_page > 1:
            row.append(InlineKeyboardButton("◀️ Prev", callback_data=f"{prefix}_page_{current_page-1}"))
        
        row.append(InlineKeyboardButton(f"{current_page}/{total_pages}", callback_data="noop"))
        
        if current_page < total_pages:
            row.append(InlineKeyboardButton("Next ▶️", callback_data=f"{prefix}_page_{current_page+1}"))
        
        keyboard.append(row)
    
    keyboard.append([InlineKeyboardButton("🔙 Back", callback_data=f"{prefix}_back")])
    
    return InlineKeyboardMarkup(keyboard)


def create_inline_keyboard(buttons: List[Dict[str, str]], rows: int = None) -> InlineKeyboardMarkup:
    """Create inline keyboard from button definitions"""
    keyboard = []
    current_row = []
    
    for button in buttons:
        current_row.append(
            InlineKeyboardButton(
                text=button['text'],
                callback_data=button.get('callback_data'),
                url=button.get('url')
            )
        )
        
        if rows and len(current_row) >= rows:
            keyboard.append(current_row)
            current_row = []
    
    if current_row:
        keyboard.append(current_row)
    
    return InlineKeyboardMarkup(keyboard)