# backend/shared/middleware/auth_middleware.py
from fastapi import Request, HTTPException
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from starlette.middleware.base import BaseHTTPMiddleware
from starlette.types import ASGIApp
import logging
from typing import Optional

from core.security import security

logger = logging.getLogger(__name__)

class AuthMiddleware(BaseHTTPMiddleware):
    """Authentication middleware"""
    
    def __init__(self, app: ASGIApp):
        super().__init__(app)
        self.security = HTTPBearer(auto_error=False)
    
    async def dispatch(self, request: Request, call_next):
        # Skip auth for public endpoints
        public_paths = [
            '/health',
            '/docs',
            '/redoc',
            '/openapi.json',
            '/api/v1/auth/login',
            '/api/v1/auth/register',
            '/webhook'
        ]
        
        if any(request.url.path.startswith(path) for path in public_paths):
            return await call_next(request)
        
        # Get token from header
        auth: Optional[HTTPAuthorizationCredentials] = await self.security(request)
        
        if auth:
            token = auth.credentials
            payload = security.decode_token(token)
            
            if payload:
                # Add user info to request state
                request.state.user = {
                    'id': payload.get('sub'),
                    'type': payload.get('user_type'),
                    'token': token
                }
            else:
                request.state.user = None
        else:
            request.state.user = None
        
        return await call_next(request)