# backend/shared/utils/formatters.py
from datetime import datetime
from typing import Any, Dict, Optional
import json

class Formatters:
    """Formatting utilities"""
    
    @staticmethod
    def format_currency(amount: float, currency: str = "INR") -> str:
        """Format amount as currency"""
        symbols = {
            "INR": "₹",
            "USD": "$",
            "EUR": "€",
            "GBP": "£"
        }
        symbol = symbols.get(currency, "₹")
        
        if amount >= 10000000:  # 1 Crore
            return f"{symbol}{amount/10000000:.2f}Cr"
        elif amount >= 100000:  # 1 Lakh
            return f"{symbol}{amount/100000:.2f}L"
        elif amount >= 1000:
            return f"{symbol}{amount/1000:.2f}K"
        else:
            return f"{symbol}{amount:.2f}"
    
    @staticmethod
    def format_number(num: float, decimals: int = 0) -> str:
        """Format large numbers with K, M, B suffixes"""
        if num >= 1e9:
            return f"{num/1e9:.{decimals}f}B"
        elif num >= 1e6:
            return f"{num/1e6:.{decimals}f}M"
        elif num >= 1e3:
            return f"{num/1e3:.{decimals}f}K"
        else:
            return f"{num:.{decimals}f}"
    
    @staticmethod
    def format_datetime(dt: datetime, format: str = "%d %b %Y %H:%M") -> str:
        """Format datetime"""
        return dt.strftime(format)
    
    @staticmethod
    def format_timespan(seconds: int) -> str:
        """Format timespan in seconds to human readable"""
        if seconds < 60:
            return f"{seconds} seconds"
        elif seconds < 3600:
            minutes = seconds // 60
            return f"{minutes} minute{'s' if minutes > 1 else ''}"
        elif seconds < 86400:
            hours = seconds // 3600
            return f"{hours} hour{'s' if hours > 1 else ''}"
        elif seconds < 2592000:
            days = seconds // 86400
            return f"{days} day{'s' if days > 1 else ''}"
        else:
            months = seconds // 2592000
            return f"{months} month{'s' if months > 1 else ''}"
    
    @staticmethod
    def format_percentage(value: float, decimals: int = 1) -> str:
        """Format as percentage"""
        return f"{value:.{decimals}f}%"
    
    @staticmethod
    def format_file_size(size_bytes: int) -> str:
        """Format file size"""
        if size_bytes < 1024:
            return f"{size_bytes} B"
        elif size_bytes < 1024 * 1024:
            return f"{size_bytes / 1024:.1f} KB"
        elif size_bytes < 1024 * 1024 * 1024:
            return f"{size_bytes / (1024 * 1024):.1f} MB"
        else:
            return f"{size_bytes / (1024 * 1024 * 1024):.2f} GB"
    
    @staticmethod
    def truncate(text: str, max_length: int = 100, suffix: str = "...") -> str:
        """Truncate text to max length"""
        if len(text) <= max_length:
            return text
        return text[:max_length - len(suffix)] + suffix
    
    @staticmethod
    def mask_sensitive(text: str, visible_chars: int = 4) -> str:
        """Mask sensitive information"""
        if len(text) <= visible_chars:
            return "*" * len(text)
        return text[:visible_chars] + "*" * (len(text) - visible_chars)
    
    @staticmethod
    def format_json(data: Any, indent: int = 2) -> str:
        """Format as pretty JSON"""
        return json.dumps(data, indent=indent, default=str)
    
    @staticmethod
    def to_camel_case(snake_str: str) -> str:
        """Convert snake_case to camelCase"""
        components = snake_str.split('_')
        return components[0] + ''.join(x.title() for x in components[1:])
    
    @staticmethod
    def to_snake_case(camel_str: str) -> str:
        """Convert camelCase to snake_case"""
        import re
        pattern = re.compile(r'(?<!^)(?=[A-Z])')
        return pattern.sub('_', camel_str).lower()