# backend/shared/utils/helpers.py
from typing import List, Any, Dict, Optional
import random
import string
import hashlib
import json
from datetime import datetime

class Helpers:
    """General helper utilities"""
    
    @staticmethod
    def chunk_list(lst: List[Any], chunk_size: int) -> List[List[Any]]:
        """Split list into chunks"""
        return [lst[i:i + chunk_size] for i in range(0, len(lst), chunk_size)]
    
    @staticmethod
    def merge_dicts(dict1: Dict, dict2: Dict, deep: bool = True) -> Dict:
        """Merge two dictionaries"""
        result = dict1.copy()
        
        for key, value in dict2.items():
            if deep and isinstance(value, dict) and key in result and isinstance(result[key], dict):
                result[key] = Helpers.merge_dicts(result[key], value)
            else:
                result[key] = value
        
        return result
    
    @staticmethod
    def generate_random_string(length: int = 8, 
                              chars: str = string.ascii_letters + string.digits) -> str:
        """Generate random string"""
        return ''.join(random.choice(chars) for _ in range(length))
    
    @staticmethod
    def generate_short_uuid() -> str:
        """Generate short UUID"""
        import uuid
        return str(uuid.uuid4())[:8]
    
    @staticmethod
    def hash_string(text: str, algorithm: str = 'sha256') -> str:
        """Hash string using specified algorithm"""
        if algorithm == 'md5':
            return hashlib.md5(text.encode()).hexdigest()
        elif algorithm == 'sha1':
            return hashlib.sha1(text.encode()).hexdigest()
        else:
            return hashlib.sha256(text.encode()).hexdigest()
    
    @staticmethod
    def safe_json_loads(data: str, default: Any = None) -> Any:
        """Safely load JSON string"""
        try:
            return json.loads(data)
        except:
            return default
    
    @staticmethod
    def parse_bool(value: Any) -> bool:
        """Parse boolean from various formats"""
        if isinstance(value, bool):
            return value
        if isinstance(value, (int, float)):
            return bool(value)
        if isinstance(value, str):
            return value.lower() in ('true', 'yes', '1', 'on')
        return False
    
    @staticmethod
    def extract_numbers(text: str) -> List[int]:
        """Extract all numbers from text"""
        import re
        return [int(num) for num in re.findall(r'\d+', text)]
    
    @staticmethod
    def pluralize(count: int, singular: str, plural: Optional[str] = None) -> str:
        """Pluralize word based on count"""
        if count == 1:
            return f"{count} {singular}"
        else:
            if plural:
                return f"{count} {plural}"
            elif singular.endswith('y'):
                return f"{count} {singular[:-1]}ies"
            else:
                return f"{count} {singular}s"
    
    @staticmethod
    def calculate_age(birth_date: datetime) -> int:
        """Calculate age from birth date"""
        today = datetime.now()
        return today.year - birth_date.year - (
            (today.month, today.day) < (birth_date.month, birth_date.day)
        )
    
    @staticmethod
    def parse_user_agent(user_agent: str) -> Dict[str, str]:
        """Parse user agent string"""
        # Basic parsing - in production use user-agents library
        result = {
            'browser': 'Unknown',
            'os': 'Unknown',
            'device': 'Unknown'
        }
        
        if 'Mobile' in user_agent:
            result['device'] = 'Mobile'
        elif 'Tablet' in user_agent:
            result['device'] = 'Tablet'
        else:
            result['device'] = 'Desktop'
        
        if 'Windows' in user_agent:
            result['os'] = 'Windows'
        elif 'Mac' in user_agent:
            result['os'] = 'macOS'
        elif 'Linux' in user_agent:
            result['os'] = 'Linux'
        elif 'Android' in user_agent:
            result['os'] = 'Android'
        elif 'iOS' in user_agent or 'iPhone' in user_agent:
            result['os'] = 'iOS'
        
        if 'Chrome' in user_agent:
            result['browser'] = 'Chrome'
        elif 'Firefox' in user_agent:
            result['browser'] = 'Firefox'
        elif 'Safari' in user_agent:
            result['browser'] = 'Safari'
        elif 'Edge' in user_agent:
            result['browser'] = 'Edge'
        
        return result