# backend/shared/utils/validators.py
import re
from typing import Any, Optional
from datetime import datetime

class Validators:
    """Validation utilities"""
    
    @staticmethod
    def validate_email(email: str) -> bool:
        """Validate email format"""
        pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
        return bool(re.match(pattern, email))
    
    @staticmethod
    def validate_phone(phone: str) -> bool:
        """Validate phone number format"""
        pattern = r'^\+?[1-9]\d{1,14}$'
        return bool(re.match(pattern, phone))
    
    @staticmethod
    def validate_telegram_id(telegram_id: Any) -> bool:
        """Validate Telegram ID"""
        try:
            tid = int(telegram_id)
            return tid > 0 and tid < 999999999999999
        except:
            return False
    
    @staticmethod
    def validate_bot_token(token: str) -> bool:
        """Validate Telegram bot token"""
        pattern = r'^\d+:[A-Za-z0-9_-]+$'
        return bool(re.match(pattern, token))
    
    @staticmethod
    def validate_username(username: str) -> bool:
        """Validate username format"""
        if not username:
            return False
        pattern = r'^[a-zA-Z0-9_]{3,32}$'
        return bool(re.match(pattern, username))
    
    @staticmethod
    def validate_password(password: str) -> tuple:
        """Validate password strength"""
        if len(password) < 8:
            return False, "Password must be at least 8 characters"
        if not re.search(r'[A-Z]', password):
            return False, "Password must contain at least one uppercase letter"
        if not re.search(r'[a-z]', password):
            return False, "Password must contain at least one lowercase letter"
        if not re.search(r'\d', password):
            return False, "Password must contain at least one number"
        return True, "Password is valid"
    
    @staticmethod
    def validate_amount(amount: Any, min_amount: float = 0, max_amount: float = 1000000) -> bool:
        """Validate amount"""
        try:
            amt = float(amount)
            return min_amount <= amt <= max_amount
        except:
            return False
    
    @staticmethod
    def validate_date(date_str: str, format: str = "%Y-%m-%d") -> bool:
        """Validate date string"""
        try:
            datetime.strptime(date_str, format)
            return True
        except:
            return False
    
    @staticmethod
    def validate_url(url: str) -> bool:
        """Validate URL"""
        pattern = r'^(https?:\/\/)?([\da-z\.-]+)\.([a-z\.]{2,6})([\/\w \.-]*)*\/?$'
        return bool(re.match(pattern, url))
    
    @staticmethod
    def validate_ip(ip: str) -> bool:
        """Validate IP address"""
        pattern = r'^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$'
        return bool(re.match(pattern, ip))
    
    @staticmethod
    def validate_key_format(key: str, min_len: int = 8, max_len: int = 50) -> bool:
        """Validate key format"""
        if not key or len(key) < min_len or len(key) > max_len:
            return False
        # Allow alphanumeric and common separators
        pattern = r'^[A-Za-z0-9\-_]+$'
        return bool(re.match(pattern, key))