# backend/tasks/broadcast_tasks.py
from celery import shared_task
from celery.utils.log import get_task_logger
from typing import List, Dict, Any, Optional
import asyncio
from datetime import datetime

from core.database import db_session
from core.models import Admin, TenantRegistry
from bot_core.bot import bot_manager
from shared.utils.helpers import chunk_list

logger = get_task_logger(__name__)

@shared_task(bind=True, max_retries=3)
def process_broadcast(self, broadcast_id: str, admin_id: int, 
                      reserve_id: str, message: str, 
                      target_user_ids: List[int], parse_mode: str = "HTML"):
    """Process broadcast delivery in background"""
    try:
        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)
        
        # Get bot instance
        bot = loop.run_until_complete(bot_manager.get_bot(admin_id))
        
        if not bot:
            logger.error(f"Bot not found for admin {admin_id}")
            return {"error": "Bot not found"}
        
        # Send in chunks to avoid rate limits
        chunk_size = 100
        results = {
            'total': len(target_user_ids),
            'success': 0,
            'failed': 0,
            'failed_users': []
        }
        
        for chunk in chunk_list(target_user_ids, chunk_size):
            # Send to each user in chunk
            for user_id in chunk:
                try:
                    # This would use asyncio.run_coroutine_threadsafe in production
                    # For simplicity, we're using run_until_complete
                    result = loop.run_until_complete(
                        bot.application.bot.send_message(
                            chat_id=user_id,
                            text=message,
                            parse_mode=parse_mode
                        )
                    )
                    results['success'] += 1
                    
                except Exception as e:
                    logger.error(f"Failed to send to {user_id}: {e}")
                    results['failed'] += 1
                    results['failed_users'].append(user_id)
            
            # Small delay between chunks
            loop.run_until_complete(asyncio.sleep(1))
        
        # Update broadcast status in database
        with db_session() as session:
            # Get tenant registry
            tenant = session.query(TenantRegistry).filter(
                TenantRegistry.reserve_id == reserve_id
            ).first()
            
            if tenant:
                # Update broadcast record in tenant database
                # This would use tenant-specific connection
                pass
        
        loop.close()
        
        logger.info(f"Broadcast {broadcast_id} completed: {results}")
        return results
        
    except Exception as e:
        logger.error(f"Broadcast failed: {e}")
        self.retry(exc=e, countdown=60)


@shared_task
def schedule_broadcast(broadcast_id: str, admin_id: int, reserve_id: str,
                       message: str, target_filters: Dict[str, Any],
                       scheduled_time: datetime):
    """Schedule broadcast for future delivery"""
    logger.info(f"Broadcast {broadcast_id} scheduled for {scheduled_time}")
    
    # In production, this would be handled by Celery Beat
    # For now, we just log it
    return {
        "broadcast_id": broadcast_id,
        "scheduled_time": scheduled_time.isoformat(),
        "status": "scheduled"
    }


@shared_task
def retry_failed_broadcasts():
    """Retry failed broadcast deliveries"""
    logger.info("Retrying failed broadcasts...")
    
    # Implementation would query failed deliveries and retry
    return {"status": "completed"}