# backend/tasks/celery_app.py
from celery import Celery
from celery.schedules import crontab
import logging
from typing import Any, Dict
import json

from config.settings import settings

logger = logging.getLogger(__name__)

# Create Celery app
celery_app = Celery(
    "telegram_bot",
    broker=settings.CELERY_BROKER_URL,
    backend=settings.CELERY_RESULT_BACKEND,
    include=[
        "tasks.broadcast_tasks",
        "tasks.key_tasks",
        "tasks.payment_tasks",
        "tasks.monitoring_tasks",
        "tasks.notification_tasks",
        "tasks.report_tasks",
        "tasks.backup_tasks",
    ]
)

# Configure Celery
celery_app.conf.update(
    task_serializer=settings.CELERY_TASK_SERIALIZER,
    result_serializer=settings.CELERY_RESULT_SERIALIZER,
    accept_content=settings.CELERY_ACCEPT_CONTENT,
    timezone=settings.CELERY_TIMEZONE,
    enable_utc=settings.CELERY_ENABLE_UTC,
    task_track_started=settings.CELERY_TASK_TRACK_STARTED,
    task_time_limit=settings.CELERY_TASK_TIME_LIMIT,
    task_soft_time_limit=settings.CELERY_TASK_SOFT_TIME_LIMIT,
    result_expires=3600,
    worker_prefetch_multiplier=1,
    worker_max_tasks_per_child=1000,
    task_acks_late=True,
    task_reject_on_worker_lost=True,
)


# Scheduled tasks
celery_app.conf.beat_schedule = {
    # Monitoring tasks
    "check-server-health": {
        "task": "tasks.monitoring_tasks.check_all_servers_health",
        "schedule": crontab(minute="*/5"),  # Every 5 minutes
    },
    "check-expiring-recharges": {
        "task": "tasks.monitoring_tasks.check_expiring_recharges",
        "schedule": crontab(hour="0", minute="0"),  # Daily at midnight
    },
    "check-low-keys": {
        "task": "tasks.key_tasks.check_low_keys_all_tenants",
        "schedule": crontab(minute="*/30"),  # Every 30 minutes
    },
    
    # Backup tasks
    "daily-database-backup": {
        "task": "tasks.backup_tasks.perform_database_backup",
        "schedule": crontab(hour="2", minute="0"),  # Daily at 2 AM
    },
    
    # Report tasks
    "send-daily-reports": {
        "task": "tasks.report_tasks.send_daily_reports",
        "schedule": crontab(hour="8", minute="0"),  # Daily at 8 AM
    },
    "send-weekly-reports": {
        "task": "tasks.report_tasks.send_weekly_reports",
        "schedule": crontab(day_of_week="monday", hour="9", minute="0"),  # Monday at 9 AM
    },
    
    # Cleanup tasks
    "cleanup-old-activities": {
        "task": "tasks.monitoring_tasks.cleanup_old_activities",
        "schedule": crontab(hour="3", minute="0"),  # Daily at 3 AM
    },
    "cleanup-expired-keys": {
        "task": "tasks.key_tasks.cleanup_expired_keys",
        "schedule": crontab(hour="4", minute="0"),  # Daily at 4 AM
    },
    
    # Notification tasks
    "send-renewal-reminders": {
        "task": "tasks.notification_tasks.send_renewal_reminders",
        "schedule": crontab(hour="10", minute="0"),  # Daily at 10 AM
    },
    
    # Analytics tasks
    "update-analytics": {
        "task": "tasks.report_tasks.update_analytics",
        "schedule": crontab(minute="*/15"),  # Every 15 minutes
    },
}


@celery_app.task(bind=True, max_retries=3)
def debug_task(self: Any) -> Dict[str, Any]:
    """Debug task to test Celery"""
    logger.info(f"Debug task executed: {self.request.id}")
    return {
        "status": "success",
        "task_id": self.request.id,
        "message": "Debug task completed"
    }


@celery_app.task(bind=True)
def health_check(self: Any) -> Dict[str, Any]:
    """Health check task"""
    return {
        "status": "healthy",
        "task_id": self.request.id,
        "timestamp": str(celery_app.now())
    }


# Task base class with common functionality
class BaseTaskWithRetry(celery_app.Task):
    """Base task class with retry and error handling"""
    
    autoretry_for = (Exception,)
    retry_kwargs = {"max_retries": 3}
    retry_backoff = True
    retry_backoff_max = 600
    retry_jitter = True
    
    def on_failure(self, exc, task_id, args, kwargs, einfo):
        """Handle task failure"""
        logger.error(f"Task {task_id} failed: {exc}")
        super().on_failure(exc, task_id, args, kwargs, einfo)


# Set custom task base
celery_app.Task = BaseTaskWithRetry