# backend/tasks/key_tasks.py
from celery import shared_task
from celery.utils.log import get_task_logger
from datetime import datetime, timedelta
from typing import Dict, Any

from core.database import db_session
from core.models import Admin, TenantRegistry
from admin_panel.services.key_service import KeyService
from shared.utils.datetime_utils import datetime_utils

logger = get_task_logger(__name__)

@shared_task
def check_low_keys_all_tenants():
    """Check low key alerts for all tenants"""
    logger.info("Checking low keys for all tenants...")
    
    with db_session() as session:
        admins = session.query(Admin).filter(
            Admin.recharge_status == 'active',
            Admin.status == 'active'
        ).all()
        
        results = []
        for admin in admins:
            result = check_tenant_low_keys.delay(admin.id, admin.reserve_id)
            results.append(result.id)
        
        return {"tenants_checked": len(admins), "task_ids": results}


@shared_task
def check_tenant_low_keys(admin_id: int, reserve_id: str):
    """Check low keys for specific tenant"""
    logger.info(f"Checking low keys for tenant {reserve_id}")
    
    try:
        # Get tenant database connection
        with db_session() as session:
            tenant = session.query(TenantRegistry).filter(
                TenantRegistry.reserve_id == reserve_id
            ).first()
            
            if not tenant:
                logger.error(f"Tenant {reserve_id} not found")
                return {"error": "Tenant not found"}
            
            # Create tenant DB URL
            db_url = f"postgresql://{tenant.database_user}:{tenant.database_password}@{tenant.database_host}:{tenant.database_port}/{tenant.database_name}"
            
            # Initialize key service
            key_service = KeyService()
            
            # Check each duration
            durations = ['1d', '3d', '7d', '30d', '60d']
            alerts = []
            
            for duration in durations:
                # Get alert config
                config = key_service.get_alert_config(reserve_id, duration, session)
                
                if config and config['enabled']:
                    # Count available keys
                    count = key_service.count_available_keys(reserve_id, duration, session)
                    
                    if count <= config['threshold']:
                        alerts.append({
                            'duration': duration,
                            'count': count,
                            'threshold': config['threshold']
                        })
                        
                        # Send notification
                        if config.get('notify_admin'):
                            send_low_key_notification.delay(
                                admin_id, reserve_id, duration, count, config['threshold']
                            )
            
            return {
                "tenant": reserve_id,
                "checked_at": datetime_utils.now().isoformat(),
                "alerts": alerts
            }
            
    except Exception as e:
        logger.error(f"Error checking low keys for {reserve_id}: {e}")
        return {"error": str(e)}


@shared_task
def send_low_key_notification(admin_id: int, reserve_id: str, 
                              duration: str, count: int, threshold: int):
    """Send low key alert notification to admin"""
    logger.info(f"Sending low key alert to admin {admin_id} for {duration} keys")
    
    # In production, this would send Telegram message
    # For now, just log it
    return {
        "admin_id": admin_id,
        "reserve_id": reserve_id,
        "duration": duration,
        "count": count,
        "threshold": threshold,
        "notification_sent": True
    }


@shared_task
def cleanup_expired_keys():
    """Mark expired keys as expired"""
    logger.info("Cleaning up expired keys...")
    
    with db_session() as session:
        admins = session.query(Admin).filter(Admin.recharge_status == 'active').all()
        
        total_expired = 0
        for admin in admins:
            try:
                # Get tenant connection
                tenant = session.query(TenantRegistry).filter(
                    TenantRegistry.admin_id == admin.id
                ).first()
                
                if tenant:
                    # Mark expired keys
                    # Implementation would update tenant database
                    pass
                    
            except Exception as e:
                logger.error(f"Error cleaning up keys for admin {admin.id}: {e}")
        
        return {"expired_keys_marked": total_expired}


@shared_task
def generate_key_batch(admin_id: int, reserve_id: str, duration: str, 
                       count: int, pattern: Dict[str, Any]):
    """Generate batch of keys"""
    logger.info(f"Generating {count} keys for {duration} for tenant {reserve_id}")
    
    try:
        from core.utils import key_generator
        
        # Generate keys
        keys = key_generator.generate_batch(
            pattern=pattern.get('pattern', settings.KEY_DEFAULT_PATTERN),
            count=count,
            prefix=pattern.get('prefix', ''),
            exclude_similar=pattern.get('exclude_similar', True),
            enable_checksum=pattern.get('checksum_enabled', False)
        )
        
        # Store in database
        # Implementation would save to tenant database
        
        return {
            "tenant": reserve_id,
            "duration": duration,
            "generated": len(keys),
            "keys": keys[:5]  # Return first 5 as sample
        }
        
    except Exception as e:
        logger.error(f"Key generation failed: {e}")
        return {"error": str(e)}