# backend/tasks/monitoring_tasks.py
from celery import shared_task
from celery.utils.log import get_task_logger
from datetime import datetime, timedelta
import logging
import psutil

from core.database import db_session
from core.models import Admin, ServerHealthLog, ServerIncident
from core.utils import datetime_utils

logger = get_task_logger(__name__)

@shared_task
def check_all_servers_health():
    """Check health of all servers"""
    logger.info("Checking all servers health...")
    
    with db_session() as session:
        admins = session.query(Admin).filter(
            Admin.status == 'active'
        ).all()
        
        for admin in admins:
            check_single_server_health.delay(admin.id, admin.reserve_id)
    
    logger.info(f"Health checks initiated for {len(admins)} servers")

@shared_task
def check_single_server_health(admin_id: int, reserve_id: str):
    """Check health of a single server"""
    logger.info(f"Checking health for server {reserve_id}")
    
    # Get system metrics
    cpu_percent = psutil.cpu_percent(interval=1)
    memory = psutil.virtual_memory()
    disk = psutil.disk_usage('/')
    
    # Create health log
    with db_session() as session:
        health_log = ServerHealthLog(
            admin_id=admin_id,
            reserve_id=reserve_id,
            cpu_usage=cpu_percent,
            ram_usage=memory.percent,
            disk_usage=disk.percent,
            bot_status='online',
            webhook_status='online',
            database_status='online',
            redis_status='online',
            response_time=234,
            checked_at=datetime_utils.now()
        )
        session.add(health_log)
        session.commit()
        
        # Check for alerts
        if cpu_percent > 80:
            create_incident.delay(
                admin_id, reserve_id, 'high_cpu',
                f"CPU usage at {cpu_percent}%", 'warning'
            )
        
        if memory.percent > 80:
            create_incident.delay(
                admin_id, reserve_id, 'high_memory',
                f"Memory usage at {memory.percent}%", 'warning'
            )
        
        if disk.percent > 90:
            create_incident.delay(
                admin_id, reserve_id, 'low_disk',
                f"Disk usage at {disk.percent}%", 'critical'
            )
    
    logger.info(f"Health check completed for {reserve_id}")

@shared_task
def create_incident(admin_id: int, reserve_id: str, incident_type: str,
                   description: str, severity: str):
    """Create server incident record"""
    logger.info(f"Creating incident for {reserve_id}: {incident_type}")
    
    with db_session() as session:
        incident = ServerIncident(
            admin_id=admin_id,
            reserve_id=reserve_id,
            incident_type=incident_type,
            severity=severity,
            start_time=datetime_utils.now(),
            description=description,
            status='open'
        )
        session.add(incident)
        session.commit()
        
        # Send notification
        send_incident_notification.delay(incident.id)
    
    return incident.id

@shared_task
def check_expiring_recharges():
    """Check for expiring admin recharges"""
    logger.info("Checking expiring recharges...")
    
    now = datetime_utils.now()
    warning_periods = [7, 3, 1]  # Days before expiry
    
    with db_session() as session:
        for days in warning_periods:
            expiry_date = now + timedelta(days=days)
            
            admins = session.query(Admin).filter(
                Admin.recharge_expiry.between(
                    expiry_date - timedelta(days=1),
                    expiry_date + timedelta(days=1)
                ),
                Admin.recharge_status == 'active'
            ).all()
            
            for admin in admins:
                send_expiry_notification.delay(admin.id, days)
                
                logger.info(f"Expiry warning sent to admin {admin.id} ({days} days)")

@shared_task
def send_expiry_notification(admin_id: int, days_left: int):
    """Send expiry notification to admin"""
    logger.info(f"Sending expiry notification to admin {admin_id} ({days_left} days)")
    
    # This would send Telegram message to admin
    pass

@shared_task
def send_incident_notification(incident_id: int):
    """Send incident notification"""
    logger.info(f"Sending incident notification for incident {incident_id}")
    
    # This would send notification to owner/admin
    pass

@shared_task
def cleanup_old_activities():
    """Clean up old activity logs"""
    logger.info("Cleaning up old activities...")
    
    retention_days = 90
    cutoff = datetime_utils.now() - timedelta(days=retention_days)
    
    # This would delete old activities from databases
    logger.info(f"Activities older than {cutoff} cleaned up")