# backend/tasks/notification_tasks.py
from celery import shared_task
from celery.utils.log import get_task_logger
import logging
import requests

from core.database import db_session
from core.models import Admin, NotificationLog
from core.utils import datetime_utils

logger = get_task_logger(__name__)

@shared_task
def send_telegram_notification(chat_id: int, message: str, parse_mode: str = 'HTML'):
    """Send Telegram notification"""
    logger.info(f"Sending Telegram notification to {chat_id}")
    
    # This would call Telegram Bot API
    # For now, just log
    logger.info(f"Telegram notification sent to {chat_id}")

@shared_task
def send_email_notification(email: str, subject: str, body: str):
    """Send email notification"""
    logger.info(f"Sending email notification to {email}")
    
    # This would send email via SMTP
    logger.info(f"Email notification sent to {email}")

@shared_task
def send_renewal_reminders():
    """Send renewal reminders to admins"""
    logger.info("Sending renewal reminders...")
    
    now = datetime_utils.now()
    reminder_days = [7, 3, 1]
    
    with db_session() as session:
        for days in reminder_days:
            expiry_date = now + timedelta(days=days)
            
            admins = session.query(Admin).filter(
                Admin.recharge_expiry.between(
                    expiry_date - timedelta(days=1),
                    expiry_date + timedelta(days=1)
                ),
                Admin.recharge_status == 'active'
            ).all()
            
            for admin in admins:
                # Create notification log
                notification = NotificationLog(
                    admin_id=admin.id,
                    notification_type='renewal_reminder',
                    channel='telegram',
                    recipient_type='admin',
                    recipient_id=admin.telegram_id,
                    subject='Renewal Reminder',
                    message=f"Your panel expires in {days} days. Renew now to avoid interruption.",
                    status='sent',
                    sent_at=datetime_utils.now()
                )
                session.add(notification)
                
                logger.info(f"Renewal reminder sent to admin {admin.id} ({days} days)")
        
        session.commit()
    
    logger.info("Renewal reminders completed")

@shared_task
def send_low_key_alert(admin_id: int, reserve_id: str, duration: str, count: int, threshold: int):
    """Send low key alert to admin"""
    logger.info(f"Sending low key alert to admin {admin_id} for {duration} keys")
    
    message = f"⚠️ **Low Key Alert**\n\n{duration} keys are running low.\nCurrent count: {count}\nThreshold: {threshold}"
    
    # This would send Telegram message
    send_telegram_notification.delay(admin_id, message)
    
    logger.info(f"Low key alert sent to admin {admin_id}")