# backend/tasks/report_tasks.py
from celery import shared_task
from celery.utils.log import get_task_logger
from datetime import datetime, timedelta
import logging
import csv
import io

from core.database import db_session
from core.models import Admin
from core.utils import datetime_utils

logger = get_task_logger(__name__)

@shared_task
def send_daily_reports():
    """Send daily activity reports to admins"""
    logger.info("Generating daily reports...")
    
    yesterday = datetime_utils.now() - timedelta(days=1)
    date_str = yesterday.strftime('%Y-%m-%d')
    
    with db_session() as session:
        admins = session.query(Admin).filter(
            Admin.recharge_status == 'active'
        ).all()
        
        for admin in admins:
            generate_admin_report.delay(admin.id, admin.reserve_id, 'daily', date_str)
    
    logger.info(f"Daily reports initiated for {len(admins)} admins")

@shared_task
def send_weekly_reports():
    """Send weekly reports to admins"""
    logger.info("Generating weekly reports...")
    
    last_week = datetime_utils.now() - timedelta(days=7)
    date_str = last_week.strftime('%Y-%m-%d')
    
    with db_session() as session:
        admins = session.query(Admin).filter(
            Admin.recharge_status == 'active'
        ).all()
        
        for admin in admins:
            generate_admin_report.delay(admin.id, admin.reserve_id, 'weekly', date_str)
    
    logger.info(f"Weekly reports initiated for {len(admins)} admins")

@shared_task
def generate_admin_report(admin_id: int, reserve_id: str, report_type: str, date_str: str):
    """Generate report for specific admin"""
    logger.info(f"Generating {report_type} report for admin {admin_id}")
    
    # This would query tenant database and generate report
    report_data = {
        'admin_id': admin_id,
        'reserve_id': reserve_id,
        'type': report_type,
        'period': date_str,
        'generated_at': datetime_utils.now().isoformat(),
        'stats': {
            'new_users': 23,
            'active_users': 456,
            'keys_sold': 45,
            'revenue': 27000
        }
    }
    
    # Save report or send via email
    logger.info(f"{report_type} report generated for admin {admin_id}")
    
    return report_data

@shared_task
def update_analytics():
    """Update analytics data for all tenants"""
    logger.info("Updating analytics...")
    
    # This would aggregate data and update analytics tables
    logger.info("Analytics updated")

@shared_task
def generate_owner_report(report_type: str):
    """Generate report for owner"""
    logger.info(f"Generating {report_type} report for owner")
    
    with db_session() as session:
        # Get system-wide stats
        total_admins = session.query(Admin).count()
        active_admins = session.query(Admin).filter(Admin.status == 'active').count()
        
        report = {
            'type': report_type,
            'generated_at': datetime_utils.now().isoformat(),
            'stats': {
                'total_admins': total_admins,
                'active_admins': active_admins,
                'total_revenue': 1500000,
                'monthly_revenue': 450000
            },
            'top_admins': [
                {'username': 'admin1', 'revenue': 250000},
                {'username': 'admin2', 'revenue': 200000}
            ]
        }
        
        logger.info(f"{report_type} report generated for owner")
        return report