// frontend/owner-dashboard/src/services/auth.js
import api from './api';

class AuthService {
  /**
   * Login user
   * @param {string} username - Username
   * @param {string} password - Password
   * @returns {Promise<Object>} Login response
   */
  async login(username, password) {
    try {
      const response = await api.post('/auth/login', { username, password });
      
      if (response.data.access_token) {
        localStorage.setItem('access_token', response.data.access_token);
        localStorage.setItem('refresh_token', response.data.refresh_token);
        localStorage.setItem('user_type', response.data.user_type);
        localStorage.setItem('user_id', response.data.user_id);
        localStorage.setItem('username', response.data.username);
      }
      
      return response.data;
    } catch (error) {
      throw error;
    }
  }

  /**
   * Logout user
   */
  logout() {
    localStorage.clear();
    window.location.href = '/login';
  }

  /**
   * Refresh access token
   * @returns {Promise<Object>} Refresh response
   */
  async refreshToken() {
    try {
      const refreshToken = localStorage.getItem('refresh_token');
      const response = await api.post('/auth/refresh', { refresh_token: refreshToken });
      
      if (response.data.access_token) {
        localStorage.setItem('access_token', response.data.access_token);
      }
      
      return response.data;
    } catch (error) {
      this.logout();
      throw error;
    }
  }

  /**
   * Get current user
   * @returns {Object} Current user
   */
  getCurrentUser() {
    return {
      id: localStorage.getItem('user_id'),
      username: localStorage.getItem('username'),
      type: localStorage.getItem('user_type'),
      token: localStorage.getItem('access_token'),
    };
  }

  /**
   * Check if user is authenticated
   * @returns {boolean} Authentication status
   */
  isAuthenticated() {
    return !!localStorage.getItem('access_token');
  }

  /**
   * Check if user is owner
   * @returns {boolean} Owner status
   */
  isOwner() {
    return localStorage.getItem('user_type') === 'owner';
  }

  /**
   * Check if user is admin
   * @returns {boolean} Admin status
   */
  isAdmin() {
    return localStorage.getItem('user_type') === 'admin';
  }

  /**
   * Get auth headers
   * @returns {Object} Auth headers
   */
  getAuthHeaders() {
    const token = localStorage.getItem('access_token');
    return token ? { Authorization: `Bearer ${token}` } : {};
  }

  /**
   * Verify 2FA code
   * @param {string} code - 2FA code
   * @returns {Promise<Object>} Verification response
   */
  async verify2FA(code) {
    return await api.post('/auth/verify-2fa', { code });
  }

  /**
   * Enable 2FA
   * @returns {Promise<Object>} Enable response
   */
  async enable2FA() {
    return await api.post('/auth/enable-2fa');
  }

  /**
   * Disable 2FA
   * @returns {Promise<Object>} Disable response
   */
  async disable2FA() {
    return await api.post('/auth/disable-2fa');
  }

  /**
   * Change password
   * @param {string} oldPassword - Old password
   * @param {string} newPassword - New password
   * @returns {Promise<Object>} Change response
   */
  async changePassword(oldPassword, newPassword) {
    return await api.post('/auth/change-password', {
      old_password: oldPassword,
      new_password: newPassword
    });
  }
}

export default new AuthService();