#!/usr/bin/env python3
# scripts/backup.py
import sys
import os
import subprocess
import datetime
import shutil
import gzip
import logging
from pathlib import Path
import argparse

sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from backend.config.settings import settings
from backend.core.utils import id_generator

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class DatabaseBackup:
    """Database backup manager"""
    
    def __init__(self):
        self.backup_dir = Path(settings.BACKUP_PATH)
        self.backup_dir.mkdir(parents=True, exist_ok=True)
        self.retention_days = settings.BACKUP_RETENTION_DAYS
        
        # Parse database URL
        db_url = settings.DATABASE_URL
        # postgresql://user:pass@host:port/dbname
        parts = db_url.replace("postgresql://", "").split("@")
        auth = parts[0].split(":")
        host_port_db = parts[1].split("/")
        host_port = host_port_db[0].split(":")
        
        self.db_user = auth[0]
        self.db_password = auth[1] if len(auth) > 1 else ""
        self.db_host = host_port[0]
        self.db_port = host_port[1] if len(host_port) > 1 else "5432"
        self.db_name = host_port_db[1]
    
    def create_backup(self) -> Path:
        """Create database backup"""
        timestamp = datetime.datetime.now().strftime("%Y%m%d_%H%M%S")
        backup_id = id_generator.generate_batch_id()
        backup_file = self.backup_dir / f"backup_{timestamp}_{backup_id}.sql"
        compressed_file = backup_file.with_suffix(".sql.gz")
        
        logger.info(f"Creating backup: {compressed_file}")
        
        # Set PGPASSWORD environment variable
        env = os.environ.copy()
        if self.db_password:
            env["PGPASSWORD"] = self.db_password
        
        # Run pg_dump
        cmd = [
            "pg_dump",
            "-h", self.db_host,
            "-p", self.db_port,
            "-U", self.db_user,
            "-d", self.db_name,
            "--format=plain",
            "--no-owner",
            "--no-acl"
        ]
        
        try:
            with open(backup_file, 'w') as f:
                result = subprocess.run(cmd, env=env, stdout=f, stderr=subprocess.PIPE, check=True)
            
            # Compress backup
            with open(backup_file, 'rb') as f_in:
                with gzip.open(compressed_file, 'wb') as f_out:
                    shutil.copyfileobj(f_in, f_out)
            
            # Remove uncompressed file
            backup_file.unlink()
            
            size_mb = compressed_file.stat().st_size / 1024 / 1024
            logger.info(f"Backup created successfully: {compressed_file} ({size_mb:.2f} MB)")
            return compressed_file
            
        except subprocess.CalledProcessError as e:
            logger.error(f"Backup failed: {e.stderr.decode()}")
            if backup_file.exists():
                backup_file.unlink()
            raise
        except Exception as e:
            logger.error(f"Backup failed: {e}")
            raise
    
    def cleanup_old_backups(self):
        """Remove backups older than retention period"""
        cutoff = datetime.datetime.now() - datetime.timedelta(days=self.retention_days)
        removed = 0
        
        for backup_file in self.backup_dir.glob("backup_*.sql.gz"):
            # Extract timestamp from filename
            # backup_20240101_120000_*.sql.gz
            try:
                parts = backup_file.stem.split('_')
                if len(parts) >= 3:
                    file_time_str = f"{parts[1]}_{parts[2]}"
                    file_time = datetime.datetime.strptime(file_time_str, "%Y%m%d_%H%M%S")
                    
                    if file_time < cutoff:
                        backup_file.unlink()
                        removed += 1
                        logger.info(f"Removed old backup: {backup_file}")
            except (IndexError, ValueError) as e:
                logger.warning(f"Could not parse timestamp from {backup_file}: {e}")
        
        logger.info(f"Removed {removed} old backups")
    
    def list_backups(self):
        """List all backups"""
        backups = sorted(self.backup_dir.glob("backup_*.sql.gz"), reverse=True)
        
        if not backups:
            logger.info("No backups found")
            return
        
        logger.info("Available backups:")
        for backup in backups:
            size = backup.stat().st_size / 1024 / 1024
            modified = datetime.datetime.fromtimestamp(backup.stat().st_mtime)
            logger.info(f"  {backup.name} ({size:.2f} MB) - {modified.strftime('%Y-%m-%d %H:%M:%S')}")
    
    def restore_backup(self, backup_file: Path):
        """Restore from backup"""
        if not backup_file.exists():
            logger.error(f"Backup file not found: {backup_file}")
            return
        
        logger.info(f"Restoring from backup: {backup_file}")
        
        # Set PGPASSWORD environment variable
        env = os.environ.copy()
        if self.db_password:
            env["PGPASSWORD"] = self.db_password
        
        # Decompress backup
        decompressed_file = backup_file.with_suffix("")
        with gzip.open(backup_file, 'rb') as f_in:
            with open(decompressed_file, 'wb') as f_out:
                shutil.copyfileobj(f_in, f_out)
        
        try:
            # Drop and recreate database
            drop_cmd = [
                "dropdb",
                "-h", self.db_host,
                "-p", self.db_port,
                "-U", self.db_user,
                "--if-exists",
                self.db_name
            ]
            logger.info(f"Running: {' '.join(drop_cmd)}")
            subprocess.run(drop_cmd, env=env, check=True)
            
            create_cmd = [
                "createdb",
                "-h", self.db_host,
                "-p", self.db_port,
                "-U", self.db_user,
                self.db_name
            ]
            logger.info(f"Running: {' '.join(create_cmd)}")
            subprocess.run(create_cmd, env=env, check=True)
            
            # Restore
            restore_cmd = [
                "psql",
                "-h", self.db_host,
                "-p", self.db_port,
                "-U", self.db_user,
                "-d", self.db_name,
                "-f", str(decompressed_file)
            ]
            logger.info(f"Running restore...")
            subprocess.run(restore_cmd, env=env, check=True)
            
            logger.info("Restore completed successfully")
            
        except subprocess.CalledProcessError as e:
            logger.error(f"Restore failed: {e}")
            raise
        finally:
            # Clean up decompressed file
            if decompressed_file.exists():
                decompressed_file.unlink()

def main():
    parser = argparse.ArgumentParser(description="Database backup manager")
    parser.add_argument("action", choices=["create", "list", "cleanup", "restore"], 
                       help="Action to perform")
    parser.add_argument("--file", help="Backup file to restore (for restore action)")
    
    args = parser.parse_args()
    
    backup_manager = DatabaseBackup()
    
    if args.action == "create":
        backup_manager.create_backup()
    elif args.action == "list":
        backup_manager.list_backups()
    elif args.action == "cleanup":
        backup_manager.cleanup_old_backups()
    elif args.action == "restore":
        if not args.file:
            logger.error("Please specify backup file with --file")
            sys.exit(1)
        backup_manager.restore_backup(Path(args.file))

if __name__ == "__main__":
    main()