#!/bin/bash
# scripts/deploy.sh
# Deployment script for Telegram Multi-Tenant Bot System

set -e

echo "=========================================="
echo "Telegram Multi-Tenant Bot System Deployment"
echo "=========================================="

# Load environment variables
if [ -f .env ]; then
    export $(cat .env | grep -v '^#' | xargs)
    echo "✓ Environment variables loaded"
else
    echo "✗ .env file not found"
    exit 1
fi

# Check prerequisites
command -v docker >/dev/null 2>&1 || { echo "✗ Docker is required but not installed"; exit 1; }
command -v docker-compose >/dev/null 2>&1 || { echo "✗ Docker Compose is required but not installed"; exit 1; }
command -v python3 >/dev/null 2>&1 || { echo "✗ Python 3 is required but not installed"; exit 1; }
command -v node >/dev/null 2>&1 || { echo "✗ Node.js is required but not installed"; exit 1; }
command -v npm >/dev/null 2>&1 || { echo "✗ npm is required but not installed"; exit 1; }

echo "✓ Prerequisites checked"

# Create necessary directories
mkdir -p logs
mkdir -p backups
mkdir -p uploads/apk
mkdir -p uploads/keys
mkdir -p monitoring/grafana/dashboards
mkdir -p monitoring/grafana/datasources
mkdir -p monitoring/logstash

echo "✓ Directories created"

# Set up Python virtual environment
echo "Setting up Python environment..."
cd backend
python3 -m venv venv
source venv/bin/activate
pip install --upgrade pip
pip install -r requirements.txt
cd ..

echo "✓ Python environment setup"

# Set up frontend
echo "Setting up frontend..."
cd frontend/owner-dashboard
npm install
npm run build
cd ../admin-dashboard
npm install
npm run build
cd ../..

echo "✓ Frontend setup"

# Initialize database
echo "Initializing database..."
cd backend
alembic upgrade head
cd ..

echo "✓ Database initialized"

# Create initial admin user
echo "Creating initial admin user..."
python3 scripts/create_admin.py

# Build and start Docker containers
echo "Building and starting Docker containers..."
docker-compose build
docker-compose up -d

echo "✓ Docker containers started"

# Wait for services to be ready
echo "Waiting for services to be ready..."
sleep 30

# Check service health
echo "Checking service health..."
curl -f http://localhost:8000/health || { echo "✗ Backend health check failed"; exit 1; }
curl -f http://localhost:3000 || { echo "✗ Frontend health check failed"; exit 1; }

echo "✓ All services are healthy"

# Set up monitoring
echo "Setting up monitoring..."
curl -X POST http://localhost:3000/api/datasources \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Prometheus",
    "type": "prometheus",
    "access": "proxy",
    "url": "http://prometheus:9090",
    "basicAuth": false
  }' || true

echo "✓ Monitoring setup"

# Create backup
echo "Creating initial backup..."
python3 scripts/backup.py

echo "✓ Initial backup created"

# Configure Telegram webhook (if in production)
if [ "$ENVIRONMENT" = "production" ]; then
    echo "Configuring Telegram webhook..."
    python3 scripts/set_webhook.py
    echo "✓ Webhook configured"
fi

echo "=========================================="
echo "Deployment completed successfully!"
echo "=========================================="
echo "Access the system at:"
echo "  Owner Dashboard: http://localhost/owner"
echo "  Admin Dashboard: http://localhost/admin"
echo "  API Documentation: http://localhost:8000/api/docs"
echo "  Grafana: http://localhost:3000 (admin/admin)"
echo "  Flower: http://localhost:5555"
echo "  RabbitMQ: http://localhost:15672 (telegram_bot/RabbitPass123!)"
echo "=========================================="
echo "Default credentials:"
echo "  Owner: owner / ChangeMe123!"
echo "  Admin: admin / Admin123!"
echo "=========================================="
echo "Please change default passwords immediately!"
echo "=========================================="