#!/usr/bin/env python3
# scripts/migrate.py
import sys
import os
import argparse
import logging
from alembic.config import Config
from alembic import command
from pathlib import Path

sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from backend.core.database import engine, Base
from backend.config.settings import settings

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class DatabaseMigrator:
    """Database migration manager"""
    
    def __init__(self):
        self.alembic_cfg = Config("alembic.ini")
    
    def create_migration(self, message: str):
        """Create a new migration"""
        logger.info(f"Creating migration: {message}")
        command.revision(self.alembic_cfg, autogenerate=True, message=message)
    
    def upgrade(self, revision: str = "head"):
        """Upgrade database to revision"""
        logger.info(f"Upgrading database to {revision}")
        command.upgrade(self.alembic_cfg, revision)
    
    def downgrade(self, revision: str = "-1"):
        """Downgrade database to revision"""
        logger.info(f"Downgrading database to {revision}")
        command.downgrade(self.alembic_cfg, revision)
    
    def show_current(self):
        """Show current revision"""
        command.current(self.alembic_cfg)
    
    def show_history(self):
        """Show migration history"""
        command.history(self.alembic_cfg)
    
    def stamp(self, revision: str):
        """Stamp database with revision"""
        logger.info(f"Stamping database with {revision}")
        command.stamp(self.alembic_cfg, revision)
    
    def init_alembic(self):
        """Initialize Alembic"""
        if not Path("alembic").exists():
            command.init(self.alembic_cfg, "alembic")
            logger.info("Alembic initialized")
        else:
            logger.info("Alembic already initialized")

def main():
    parser = argparse.ArgumentParser(description="Database migration tool")
    parser.add_argument("action", choices=["init", "create", "upgrade", "downgrade", 
                                          "current", "history", "stamp"],
                       help="Migration action")
    parser.add_argument("--message", "-m", help="Migration message (for create)")
    parser.add_argument("--revision", "-r", help="Revision (for upgrade/downgrade/stamp)",
                       default="head")
    
    args = parser.parse_args()
    
    migrator = DatabaseMigrator()
    
    if args.action == "init":
        migrator.init_alembic()
    elif args.action == "create":
        if not args.message:
            logger.error("Message required for create action")
            sys.exit(1)
        migrator.create_migration(args.message)
    elif args.action == "upgrade":
        migrator.upgrade(args.revision)
    elif args.action == "downgrade":
        migrator.downgrade(args.revision)
    elif args.action == "current":
        migrator.show_current()
    elif args.action == "history":
        migrator.show_history()
    elif args.action == "stamp":
        migrator.stamp(args.revision)

if __name__ == "__main__":
    main()