#!/usr/bin/env python3
# scripts/set_webhook.py
import asyncio
import argparse
import sys
import os
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

import aiohttp
import logging
from backend.config.settings import settings
from backend.core.database import SessionLocal
from backend.core.models import Admin

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

async def set_webhook(bot_token: str, webhook_url: str, secret_token: str = None):
    """Set webhook for a bot"""
    url = f"https://api.telegram.org/bot{bot_token}/setWebhook"
    
    payload = {
        "url": webhook_url,
        "max_connections": 100,
        "allowed_updates": ["message", "callback_query", "inline_query", "chosen_inline_result"]
    }
    
    if secret_token:
        payload["secret_token"] = secret_token
    
    async with aiohttp.ClientSession() as session:
        async with session.post(url, json=payload) as response:
            result = await response.json()
            if result.get("ok"):
                logger.info(f"Webhook set successfully for bot")
                logger.info(f"Response: {result}")
            else:
                logger.error(f"Failed to set webhook: {result}")
            return result

async def get_webhook_info(bot_token: str):
    """Get current webhook info"""
    url = f"https://api.telegram.org/bot{bot_token}/getWebhookInfo"
    
    async with aiohttp.ClientSession() as session:
        async with session.get(url) as response:
            result = await response.json()
            if result.get("ok"):
                logger.info(f"Webhook info: {result}")
            else:
                logger.error(f"Failed to get webhook info: {result}")
            return result

async def delete_webhook(bot_token: str):
    """Delete webhook"""
    url = f"https://api.telegram.org/bot{bot_token}/deleteWebhook"
    
    async with aiohttp.ClientSession() as session:
        async with session.get(url) as response:
            result = await response.json()
            if result.get("ok"):
                logger.info("Webhook deleted successfully")
            else:
                logger.error(f"Failed to delete webhook: {result}")
            return result

async def set_all_webhooks(webhook_base_url: str):
    """Set webhooks for all active bots"""
    db = SessionLocal()
    try:
        admins = db.query(Admin).filter(Admin.status == 'active').all()
        
        for admin in admins:
            webhook_url = f"{webhook_base_url}/{admin.bot_token}"
            logger.info(f"Setting webhook for {admin.bot_username}")
            await set_webhook(admin.bot_token, webhook_url, settings.TELEGRAM_WEBHOOK_SECRET)
            await asyncio.sleep(1)  # Rate limiting
    
    finally:
        db.close()

async def main():
    parser = argparse.ArgumentParser(description="Manage Telegram webhooks")
    parser.add_argument("action", choices=["set", "get", "delete", "set-all"], 
                       help="Action to perform")
    parser.add_argument("--token", help="Bot token (for single bot)")
    parser.add_argument("--url", help="Webhook URL base", 
                       default=settings.TELEGRAM_WEBHOOK_URL)
    
    args = parser.parse_args()
    
    if args.action == "set-all":
        if not args.url:
            logger.error("Webhook URL required for set-all")
            sys.exit(1)
        await set_all_webhooks(args.url)
    elif args.action == "set":
        if not args.token or not args.url:
            logger.error("Token and URL required for set")
            sys.exit(1)
        await set_webhook(args.token, f"{args.url}/{args.token}", 
                         settings.TELEGRAM_WEBHOOK_SECRET)
    elif args.action == "get":
        if not args.token:
            logger.error("Token required for get")
            sys.exit(1)
        await get_webhook_info(args.token)
    elif args.action == "delete":
        if not args.token:
            logger.error("Token required for delete")
            sys.exit(1)
        await delete_webhook(args.token)

if __name__ == "__main__":
    asyncio.run(main())