# tests/conftest.py
import pytest
from sqlalchemy import create_engine
from sqlalchemy.orm import sessionmaker
from fastapi.testclient import TestClient

from backend.main import app
from backend.core.database import Base, get_db
from backend.core.models import Owner, Admin
from backend.core.security import security

TEST_DATABASE_URL = "sqlite:///./test.db"

@pytest.fixture
def test_db():
    engine = create_engine(TEST_DATABASE_URL, connect_args={"check_same_thread": False})
    TestingSessionLocal = sessionmaker(autocommit=False, autoflush=False, bind=engine)
    
    Base.metadata.create_all(bind=engine)
    
    db = TestingSessionLocal()
    try:
        yield db
    finally:
        db.close()
        Base.metadata.drop_all(bind=engine)

@pytest.fixture
def client(test_db):
    def override_get_db():
        try:
            yield test_db
        finally:
            pass
    
    app.dependency_overrides[get_db] = override_get_db
    yield TestClient(app)
    app.dependency_overrides.clear()

@pytest.fixture
def test_owner(test_db):
    owner = Owner(
        telegram_id=123456789,
        username="test_owner",
        first_name="Test",
        last_name="Owner",
        password_hash=security.get_password_hash("Test123!"),
        email="test@example.com"
    )
    test_db.add(owner)
    test_db.commit()
    test_db.refresh(owner)
    return owner

@pytest.fixture
def test_admin(test_db, test_owner):
    admin = Admin(
        owner_id=test_owner.id,
        telegram_id=987654321,
        admin_username="test_admin",
        admin_first_name="Test",
        admin_last_name="Admin",
        password_hash=security.get_password_hash("Admin123!"),
        reserve_id="TEST-001",
        bot_name="Test Bot",
        bot_username="@test_bot",
        bot_token="123456:ABCdefGHIjklMNOpqrsTUVwxyz",
        status="active",
        recharge_status="active"
    )
    test_db.add(admin)
    test_db.commit()
    test_db.refresh(admin)
    return admin

@pytest.fixture
def owner_token(test_owner):
    return security.create_access_token(
        data={"sub": str(test_owner.id), "user_type": "owner"}
    )

@pytest.fixture
def admin_token(test_admin):
    return security.create_access_token(
        data={"sub": str(test_admin.id), "user_type": "admin"}
    )